/*
This software was written by Stefan Hollos and is Copyright 2008 by Exstrom
Laboratories LLC.  This software is part of the book "Signals from the
Subatomic World: How to Build a Proton Precession Magnetometer", and is
covered by the book's copyright.
*/

#include <windows.h>
#include <conio.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>

#define __AL_WIN32
#include "..\include\adlib.h"
#include "..\include\adlerr.h"

#define _export 
#define ADLIB_CON_FILE_PATH ".\\adlib.con"

#define MAXLINESIZE 128
#define MAX_STATES  10

#define CMD_SET_PROGRAM    1
#define CMD_GET_PROGRAM    2
#define CMD_RUN_PROGRAM    3
#define CMD_LOOP_PROGRAM   4
#define CMD_STOP_PROGRAM   5

//Microcontroller clock frequency in units of Hz
#define FCLOCK_UC  10000000

HANDLE OpenSerialPort( char *portname ,DWORD baudrate );

/****************************************************************/

HANDLE OpenSerialPort( char *portname ,DWORD baudrate )
{
  HANDLE hCom;
  DWORD dwError;
  DCB dcb;
  BOOL fSuccess;

  hCom = CreateFile( portname,
    GENERIC_READ | GENERIC_WRITE,
    0,    // comm devices must be opened w/exclusive-access 
    NULL, // no security attributes 
    OPEN_EXISTING, // comm devices must use OPEN_EXISTING 
    0,    // not overlapped I/O 
    NULL  // hTemplate must be NULL for comm devices 
    );

  if( hCom == INVALID_HANDLE_VALUE )
  {
    dwError = GetLastError();
    return( NULL );
  }

// Omit the call to SetupComm to use the default queue sizes.

  fSuccess = GetCommState( hCom, &dcb );
  if( !fSuccess ) return( NULL );

// Fill in the DCB: 8 data bits, no parity, 1 stop bit. 

  dcb.BaudRate = baudrate;
  dcb.ByteSize = 8;
  dcb.Parity = NOPARITY;
  dcb.StopBits = ONESTOPBIT;

  fSuccess = SetCommState( hCom, &dcb );
  if( !fSuccess ) return( NULL );

  return( hCom );
}

/****************************************************************/

int main( int argc, char *argv[] )
{
  char *portname = "COMX";
  int pn;   // port number
  DWORD br; // baud rate
  HANDLE hCom;
  DWORD nbr;
  unsigned char ch;
  unsigned int nstates;
  unsigned int ntic;

  FILE *fp;
  char OutFile[64];
  char linebuff[MAXLINESIZE];
  char DevName[32];
  char intype;
  char polar;
  int  chnum, gain, np, ix;
  long i,j,ns;
  double sps, asps;
  double daqd,pl,pd;
  double *fbuffer;

  CHANLIST ChanList;
  ERRNUM errnum;
  LPBUFFSTRUCT lpbuff = NULL;
  LHLD  lhldAdc0;  /* ADLIB device handle */

  clock_t start;
  double  duration;

  if( argc < 3 )
  {
    printf("\nUsage: adl paramfile pn\n");
    printf("  paramfile = parameter file name\n");
    printf("  pn = serial port number: 1,2,3,4\n");
    return(-1);
  }


// Open the serial port connection to the micro-controler

  pn = atoi( argv[2] );
  if( (pn < 1) || (pn > 4) )
  {
    perror( "Portnumber must be in the range [1,4]\n" );
    return( -1 );
  }
  portname[3] = '0' + pn;

  br = 9600; //The switcher can only communicate using 9600 baud

  hCom = OpenSerialPort( portname, br );
  if( hCom == NULL )
  {
    perror( "Could not open port\n");
    return( -1 );
  }

  printf( "\nSerial port opened\n" );
  printf( "  Port = %s\n", portname );
  printf( "  Baud = %d\n\n", br );

// Open and read the parameter file

  fp = fopen(argv[1], "r");
  if( fp == 0 )
  {
    perror( "Unable to open parameter file" );
    return(-1);
  }

  printf( "Reading parameter file: %s\n\n", argv[1] );

  fgets( linebuff, MAXLINESIZE, fp );
  //Device name (32 char max) [ADAC5501]
  sscanf( linebuff, "%32s", DevName );
  fgets( linebuff, MAXLINESIZE, fp );
  //Input type [D]ifferential or [S]ingle ended
  sscanf( linebuff, "%c", &intype );
  fgets( linebuff, MAXLINESIZE, fp );
  //Channel number [0-7] (Diff) or [0-15] (SE)
  sscanf( linebuff, "%d", &chnum );
  fgets( linebuff, MAXLINESIZE, fp );
  //Polarity [U]nipolar or [B]ipolar
  sscanf( linebuff, "%c", &polar );
  fgets( linebuff, MAXLINESIZE, fp );
  //Gain [1,2,4,8,16,32,64]
  sscanf( linebuff, "%d", &gain );
  fgets( linebuff, MAXLINESIZE, fp );
  //Samples per second
  sscanf( linebuff, "%lf", &sps );
  fgets( linebuff, MAXLINESIZE, fp );
  //Number of samples
  sscanf( linebuff, "%d", &ns );
  fgets( linebuff, MAXLINESIZE, fp );
  //Pulse length in seconds
  sscanf( linebuff, "%lf", &pl );
  fgets( linebuff, MAXLINESIZE, fp );
  //Daq trigger delay in seconds
  sscanf( linebuff, "%lf", &daqd );
  fgets( linebuff, MAXLINESIZE, fp );
  //Number of pulses
  sscanf( linebuff, "%d", &np );
  fgets( linebuff, MAXLINESIZE, fp );
  //Pulse delay
  sscanf( linebuff, "%lf", &pd );
  fgets( linebuff, MAXLINESIZE, fp );
  //Output file name (64 char max)
  sscanf( linebuff, "%64s", OutFile );
  fclose(fp);

  printf("Read the following parameters\n\n");
  printf("Device name: %s\n", DevName);
  printf("Input type: %s\n",intype=='D' ? "differential":"single ended");
  printf("Channel number: %d\n", chnum);
  printf("Polarity: %s\n", polar == 'U' ? "unipolar" : "bipolar");
  printf("Gain: %d\n", gain);
  printf("Samples per second: %f\n", sps);
  printf("Number of samples: %d\n", ns);
  printf("Pulse length (sec): %lf\n", pl);
  printf("Daq trig. delay (sec): %lf\n", daqd);
  printf("Number of pulses: %d\n", np);
  printf("Pulse delay: %lf\n", pd);
  printf("Output file name: %s\n", OutFile);

// Allocate the buffer

  fbuffer = (double *)calloc( ns, sizeof(double) );
  if( fbuffer == 0 )
  {
    printf( "Unable to allocate data buffer\n" );
    return( -1 );
  }

// Configure the pulser

  printf("\nConfiguring the pulser\n");

  if( daqd < 0 ) daqd *= -1;
  if( daqd > 6.7 ) daqd = 6.7;
  nstates = daqd > 0 ? 3 : 2;
  if( pl > 13.0 ) pl = 13.0;
  if( pl > 6.7108864 ) ++nstates;

  printf("Number of states = %1u\n", nstates);

  ch = (char)CMD_SET_PROGRAM;
  WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );

  ch = nstates & 0x000000FF;
  WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );

  ch = 0;  // first state turns current switch on
  WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );

  ntic = (unsigned int)(FCLOCK_UC*pl/1024.0);
  if( ntic <= 65536 )
  {
    ntic = 65536 - ntic;
    ch = ntic & 0x000000FF;
    WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );
    ch = (ntic & 0x0000FF00) >> 8;
    WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );
    printf("state,time = 0, %1.7lf (%1u)\n", pl, 65536-ntic);
  }
  else
  {
    ch = 0;
    WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );
    WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );
    printf("state,time = 0, 6.7108864 (65536)\n");

    ch = 0; // One more state with current switch on
    WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );
    ntic = 2*65536 - ntic;
    ch = ntic & 0x000000FF;
    WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );
    ch = (ntic & 0x0000FF00) >> 8;
    WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );
    printf("state,time = 0, %1.7lf (%1u)\n", pl-6.7108864, 65536-ntic);
  }

  if( daqd > 0 )
  {
    ch = 1;  // this state turns current switch off
    WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );
    ntic = (unsigned int)(FCLOCK_UC*daqd/1024.0);
    ntic = 65536 - ntic;
    ch = ntic & 0x000000FF;
    WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );
    ch = (ntic & 0x0000FF00) >> 8;
    WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );
    printf("state,time = 1, %1.7lf (%1u)\n", daqd, 65536-ntic);
  }

  ch = 129;  // this state triggers the data acquisition
  WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );
  ntic = 65534;
  ch = ntic & 0x000000FF;
  WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );
  ch = (ntic & 0x0000FF00) >> 8;
  WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );
  printf("state,time = 129, 0.0001024 (1)\n");

// Configure the data acquisition system

  printf("\nConfiguring the data acquisition system\n");

  errnum = AL_LoadEnvironment(ADLIB_CON_FILE_PATH);
  if (errnum < 0)
  {
    printf("LoadEnvironment: ADLIB error#  %ld", errnum);
    return(errnum);
  }

  lhldAdc0 = AL_AllocateDevice("ADC0", 0);
  if (lhldAdc0 < 0)
  {
    printf("AllocateDevice: ADLIB error#  %ld", (ERRNUM)lhldAdc0);
    return(errnum);
  }

  AL_SetTriggerMode(lhldAdc0, "POST_TRIG");
  AL_SetTriggerSource(lhldAdc0, "EXTERNAL");
  AL_SetTrigSourceSignal(lhldAdc0, "RISING_EDGE");
  AL_SetClockSource(lhldAdc0, "INTERNAL");
  AL_SetClockRate(lhldAdc0, sps, AL_HERTZ);
  AL_GetActualClkRate(lhldAdc0, &asps);
  printf( "Requested samples per second = %lf\n", sps );
  printf( "Actual samples per second = %lf\n", asps );

  ChanList.lType = ARRAY_LIST;
  ChanList.lNumElements = 1;
  ChanList.alChannelList[0] = chnum;
  ChanList.alChannelList[1] = gain;
  errnum= AL_SetChannelList(lhldAdc0, &ChanList);
  if (errnum < 0)
  {
    printf("SetChannelList: ADLIB error#  %ld", errnum);
    return(errnum);
  }

  AL_SetDataOffsetGlobal(lhldAdc0,(polar=='U' ? "UNIPOLAR":"BIPOLAR"));
  AL_SetBufferSize(lhldAdc0, ns);
  AL_SetNumOfBuffers(lhldAdc0, 1);
  AL_SetAutoInitBuffers(lhldAdc0, YES);
  AL_SetBufferDoneHandler(lhldAdc0, AL_CHECK_BUFFER);          
	
  errnum = AL_InitDevice(lhldAdc0);
  if (errnum < 0)
  {
    printf("InitDevice: ADLIB error#  %ld", errnum);
    return(errnum);
  }

  printf("\nData acquisition system has been initialized\n");
  printf("\nStarting pulse sequence\n\n");

  for( i=0; i<np; ++i )
  {
    ch = (char)CMD_RUN_PROGRAM;
    WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );

    printf("Executing pulse %d\n", i);

    lpbuff = NULL;
    AL_StartDevice(lhldAdc0);
    while( lpbuff == NULL )
      lpbuff = AL_GetDoneBuffPtr(lhldAdc0);
    AL_StopDevice(lhldAdc0);

    printf("Saving data\n");

    for( j=0; j<ns; ++j )
    {
      ix = (int)(lpbuff->hpwBufferLinear[j]) & 0x0FFF;
      if( ix & 0x0800 ) ix -= 4096;
      fbuffer[j] += ix;
    }

    AL_ClearBufferDoneFlag(lhldAdc0, lpbuff->dwBuffNum);

    printf("  Starting %lf sec. pulse delay\n", pd);
    start = clock();
    do
    {
      duration = (double)(clock() - start) / CLOCKS_PER_SEC;
    }while( duration < pd );
    printf("  Pulse delay finished\n\n");
  }

  printf("\nPulse sequence finished\n\n");

  for( j=0; j<ns; ++j )
    fbuffer[j] /= (double)np;

  fp = fopen( OutFile, "w" );
  if( fp != NULL )
  {
   printf("Saving data to file: %s\n",OutFile);
   fprintf(fp,"# %20s:Input type\n",intype=='D'?"differential":"single ended");
   fprintf(fp,"# %20d:Channel number\n",chnum);
   fprintf(fp,"# %20s:Polarity\n",polar=='U'?"unipolar":"bipolar");
   fprintf(fp,"# %20d:Gain\n",gain);
   fprintf(fp,"# %20f:Requested samples per second\n",sps);
   fprintf(fp,"# %20f:Actual samples per second\n",asps);
   fprintf(fp,"# %20d:Number of samples\n",ns);
   fprintf(fp,"# %20lf:Pulse length\n",pl);
   fprintf(fp,"# %20lf:Daq trigger delay\n",daqd);
   fprintf(fp,"# %20d:Number of pulses\n",np);
   fprintf(fp,"# %20lf:Pulse delay\n",pd);

   for( i=0; i<ns; ++i )
     fprintf( fp, "%lf\n", fbuffer[i]*10.0/2048.0/(double)gain );
   fclose( fp );
  }
  else
    printf( "Could not open file: %s\n", OutFile );

  AL_ReleaseDevice(lhldAdc0);
  AL_ReleaseEnvironment();
  free( fbuffer );
  return( 0 );
}
