; This software was written by Stefan Hollos and is Copyright 2008 by Exstrom
; Laboratories LLC.  This software is part of the book "Signals from the
; Subatomic World: How to Build a Proton Precession Magnetometer", and is
; covered by the book's copyright.

.nolist
.include "2313def.inc"
.list

.equ  MAX_STATES         = 10

.equ  CMD_SET_PROGRAM    = 1
.equ  CMD_GET_PROGRAM    = 2
.equ  CMD_RUN_PROGRAM    = 3
.equ  CMD_LOOP_PROGRAM   = 4
.equ  CMD_STOP_PROGRAM   = 5

;For the magnum port b pin 0 controls the current switch
;For the magnum port b pin 1 controls the daq board

.def  bloop=r23   ;controls looping 0 = no loop, 1 = loop.
.def  istate=r24  ;The program state counter
.def  nstate=r25  ;The number of program states

.cseg

.org $000
  rjmp  RESET
.org $005
  rjmp TIM1_OVF      ;Timer1 Overflow interrupt handler
.org $007
  rjmp UART_RXC      ;UART RX Complete interrupt handler

;******************************************************
; Interrupt Handler: RESET
; Description: Execution always starts here on power up
;   or reset. There is no return from here.
	
RESET:

;Initialize the ports
  ser r16
  out ddrb,r16        ;make all portb pins outputs
  out ddrd,r16        ;make all portd pins outputs
                      ;  (not used for magnum)
  clr r16
  out portb,r16       ;set all portb pins low
  out portd,r16       ;set all portd pins low
                      ;  (not used for magnum)

;For the magnum bit 0 of port B controls the current switch. A low on this
;bit will turn the switch on, applying current to the coil, and a high will
;turn it off. The bit should therefor come up high so that the current is
;initially turned off.

  sbi portb,0         ;set initial current switch position to off for magnum

;Initialize stack pointer. The stack grows toward $00.
;The stack is offset from the end of the SRAM by the number of bytes needed
;to store the states. Three bytes are required to store each state.
;MAX_STATES is defined above. RAMEND is defined in: 2313def.inc

  ldi  r16,low(RAMEND-3*MAX_STATES)
  out  SPL,r16

;Initialize the general interrupt mask register to enable/disable external
;interrupts. There are two external interrupts int0 and int1 and they are
;enabled/disabled as follows:
; $00=no interrupts enabled, $40=int0, $80=int1, $C0=int0+int1
;See AT90S2313 manual p.24 for more details

  ldi r16,$00      ;no external interrupts are used in the magnum
  out GIMSK,r16

;Initialize the MCU control register.
;This can be used to control the way external interrupts are triggered,
;enable SRAM, wait states and sleep mode.
;See AT90S2313 manual p.26 for more details.
  ldi r16,$00
  out MCUCR,r16

;Initialize the timer/counter interrupt mask register.
;$80 enables just the timer1(16 bit) overflow interrupt
;See AT90S2313 manual p.24-25 for more details.
  ldi r16,$80
  out TIMSK,r16

;Initialize the timer/counter control registers.
;$00 stops the timer/counter
;See AT90S2313 manual p.31-33 for more details.
  clr r16
  out TCCR1A,r16
  out TCCR1B,r16

;Initialize the UART control register to enable transmit, receive, and
;receive interrupt. See AT90S2313 manual p.43 for more details.
  ldi  r16,$98  ;$98 = 1001 1000
  out  UCR,r16

;Initialize the UART Baud rate register using formula:
;   UBRR = fck/(16*BAUD) - 1
;fck = 10,000,000 (10 MHz), BAUD =  9600, UBRR = 64 (%error = .16)
;fck =  4,000,000 ( 4 MHz), BAUD = 19200, UBRR = 12 (%error = .16)
;See AT90S2313 manual p.44-45 for more details.
  ldi r16,64
  out UBRR,r16

;Initialize the registers
  ldi bloop,$00
  ldi istate,$00
  ldi nstate,$00

  sei  ;set the global interrupt flag

IDLE:
  nop
  rjmp IDLE

;*****************************************************************
; Interrupt Handler: TIM1_OVF
; Description: Takes care of Timer1 overflow events.
; Note: 6 clock cycles between the event and the execution of this code.

TIM1_OVF:
  clr r17
  out TCCR1B,r17  ;stop the timer

  cp istate,nstate
  brne START_NEXT_STATE
  cpi bloop,1
  breq RESET_STATES
  reti

RESET_STATES:
  clr istate
  clr r31
  ldi r30,low(RAMEND)+1

START_NEXT_STATE:
  ld r17,-Z  ;load value
  ld r18,-Z  ;load timer low byte
  ld r19,-Z  ;load timer high byte
  out portb,r17  ;set portb
  out TCNT1H,r19
  out TCNT1L,r18
  inc istate

  ldi r17,5
  out TCCR1B,r17  ;start the timer at 1024 clocks/tick
  reti

;***********************************************************************
; Interrupt Handler: UART_RXC
; Description: Takes care of UART data receive events

UART_RXC:
  in r16,UDR      ;read byte from UART data register

  clr r17
  out TCCR1B,r17  ;stop the timer

CASE0_UART:
  cpi r16,CMD_SET_PROGRAM
  brne CASE1_UART

  SET_NSTATE:
    sbis USR,RXC
    rjmp SET_NSTATE
  in nstate,UDR

  clr istate
  clr r31
  ldi r30,low(RAMEND)+1

  SET_STATE_BYTE:
    sbis USR,RXC
    rjmp SET_STATE_BYTE
  in r17,UDR

  SET_STATE_TIME0:
    sbis USR,RXC
    rjmp SET_STATE_TIME0
  in r18,UDR

  SET_STATE_TIME1:
    sbis USR,RXC
    rjmp SET_STATE_TIME1
  in r19,UDR

  st -Z,r17  ;store value
  st -Z,r18  ;store timer low byte
  st -Z,r19  ;store timer high byte

  inc istate
  cp istate,nstate
  brne SET_STATE_BYTE
  reti

CASE1_UART:
  cpi r16,CMD_GET_PROGRAM
  brne CASE2_UART

  out UDR,nstate

  clr istate
  clr r31
  ldi r30,low(RAMEND)+1

  GET_STATE:
    ld r17,-Z  ;load value
    ld r18,-Z  ;load timer low byte
    ld r19,-Z  ;load timer high byte

    GET_STATE_BYTE:
      sbis USR,UDRE
      rjmp GET_STATE_BYTE
    out UDR,r17

    GET_STATE_TIME0:
      sbis USR,UDRE
      rjmp GET_STATE_TIME0
    out UDR,r18

    GET_STATE_TIME1:
      sbis USR,UDRE
      rjmp GET_STATE_TIME1
    out UDR,r19

    inc istate
    cp istate,nstate
    brne GET_STATE

  reti

CASE2_UART:
  cpi r16,CMD_LOOP_PROGRAM
  brne CASE3_UART

  cpi nstate,0
  brne SET_LOOP_FLAG
  reti

  SET_LOOP_FLAG:
    ldi bloop,1
    rjmp RUN_PROGRAM

CASE3_UART:
  cpi r16,CMD_RUN_PROGRAM
  brne CASE4_UART

  clr bloop

  cpi nstate,0
  brne RUN_PROGRAM
  reti

  RUN_PROGRAM:
    clr istate
    clr r31
    ldi r30,low(RAMEND)+1

    ld r17,-Z  ;load value
    ld r18,-Z  ;load timer low byte
    ld r19,-Z  ;load timer high byte

    out portb,r17   ;set portb
    out TCNT1H,r19  ;set timer high byte
    out TCNT1L,r18  ;set timer low byte
    inc istate

    ldi r17,5
    out TCCR1B,r17  ;start the timer at 1024 clocks/tick
    reti

CASE4_UART:
  cpi r16,CMD_STOP_PROGRAM
  brne CASE5_UART

  clr bloop

CASE5_UART:
  reti
