/*
 *                            COPYRIGHT
 *
 *  mc - manual control program for the Magnum pulse controller.
 *  Copyright (C) 2008 Exstrom Laboratories LLC
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  A copy of the GNU General Public License is available on the internet at:
 *
 *  http://www.gnu.org/copyleft/gpl.html
 *
 *  or you can write to:
 *
 *  The Free Software Foundation, Inc.
 *  675 Mass Ave
 *  Cambridge, MA 02139, USA
 *
 *  You can contact Exstrom Laboratories LLC via Email at:
 *
 *  stefan(AT)exstrom.com
 *
 *  or you can write to:
 *
 *  Exstrom Laboratories LLC
 *  P.O. Box 7651
 *  Longmont, CO 80501, USA
 *
 */

/*
This program allows individual commands to be sent to the
Magnum pulse controller.
*/

#include <windows.h>
#include <conio.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>

#define MAXLINESIZE 128
#define MAX_STATES  10

#define CMD_SET_PROGRAM    1
#define CMD_GET_PROGRAM    2
#define CMD_RUN_PROGRAM    3
#define CMD_LOOP_PROGRAM   4
#define CMD_STOP_PROGRAM   5

// Microcontroller clock frequency in units of Hz
// 4000000 for the programmer board
//10000000 for the magnum board
#define FCLOCK_UC  10000000.0

HANDLE OpenSerialPort( char *portname ,DWORD baudrate );
void SetProgram(void);
void GetProgram(void);
void RunProgram(void);
void LoopProgram(void);
void StopProgram(void);

//Global variables

HANDLE hCom;

/****************************************************************************/

HANDLE OpenSerialPort( char *portname ,DWORD baudrate )
{
  HANDLE hCom;
  DWORD dwError;
  DCB dcb;
  BOOL fSuccess;

  hCom = CreateFile( portname,
    GENERIC_READ | GENERIC_WRITE,
    0,    // comm devices must be opened w/exclusive-access 
    NULL, // no security attributes 
    OPEN_EXISTING, // comm devices must use OPEN_EXISTING 
    0,    // not overlapped I/O 
    NULL  // hTemplate must be NULL for comm devices 
    );

  if( hCom == INVALID_HANDLE_VALUE )
  {
    dwError = GetLastError();
    return( NULL );
  }

// Omit the call to SetupComm to use the default queue sizes.

  fSuccess = GetCommState( hCom, &dcb );
  if( !fSuccess ) return( NULL );

// Fill in the DCB: 8 data bits, no parity, 1 stop bit. 

  dcb.BaudRate = baudrate;
  dcb.ByteSize = 8;
  dcb.Parity = NOPARITY;
  dcb.StopBits = ONESTOPBIT;

  fSuccess = SetCommState( hCom, &dcb );
  if( !fSuccess ) return( NULL );

  return( hCom );
}

/****************************************************************************/

void SetProgram(void)
{
  char cbuff[64];
  FILE *fp;
  DWORD nbr;
  unsigned char ch;
  unsigned int i,ns,ist;
  unsigned int ntic;
  double ftime;

  printf("Enter file name => ");
  scanf("%64s", cbuff);
  printf("\n");

// Open and read the parameter file
  fp = fopen(cbuff, "r");
  if( fp == 0 )
  {
    perror( "Unable to open file" );
    return;
  }

  fscanf( fp, "%u", &ns ); //read number of states
  printf("Number of states = %1u\n\n", ns);

  ch = (char)CMD_SET_PROGRAM;
  WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );

  ch = ns & 0x000000FF;
  WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );

  for(i=0; i<ns; ++i)
  {
    fscanf(fp, "%u", &ist);
    fscanf(fp, "%lf", &ftime );
    printf("state %1u = %1u\ntime = %lf\n", i, ist, ftime);

    ch = ist & 0x000000FF;
    WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );

    ntic = (unsigned int)(FCLOCK_UC*ftime/1024.0);
    ntic = 65536 - ntic;
    printf("ntic = %1u\n", ntic);
    printf("low=%1u, high=%1u\n\n", ntic & 0x000000FF,(ntic & 0x0000FF00) >> 8 );

    ch = ntic & 0x000000FF;
    WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );
    ch = (ntic & 0x0000FF00) >> 8;
    WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );
  }

  fclose(fp);
  return;
}

/****************************************************************************/

void GetProgram(void)
{
  DWORD nbr;
  unsigned char ch;
  unsigned int i,ns,ist;
  unsigned int ntic;
  double ftime;

  ch = (char)CMD_GET_PROGRAM;
  WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );

  ns = 0;
  ReadFile( hCom, (LPVOID)(&ns), 1, (LPDWORD)(&nbr), NULL );
  printf( "Number of states = %u\n", ns );
   
  for(i=0; i<ns; ++i)
  {
    ist=0;
    ReadFile( hCom, (LPVOID)(&ist), 1, (LPDWORD)(&nbr), NULL );
    printf("state %1u = %u\n", i, ist);

    ntic=0;
    ReadFile( hCom, (LPVOID)(&ntic), 2, (LPDWORD)(&nbr), NULL );
    printf("ntic = %u\n", ntic);
    printf("low=%1u, high=%1u\n", ntic & 0x000000FF,(ntic & 0x0000FF00) >> 8 );
    ftime = (double)(65536 - ntic)*1024.0/FCLOCK_UC;
    printf("time = %1.4lf\n\n",ftime);
  }

  return;
}

/****************************************************************************/

void RunProgram(void)
{
  DWORD nbr;
  unsigned char ch;

  ch = (char)CMD_RUN_PROGRAM;
  WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );

  return;
}

/****************************************************************************/

void LoopProgram(void)
{
  DWORD nbr;
  unsigned char ch;

  ch = (char)CMD_LOOP_PROGRAM;
  WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );

  return;
}

/****************************************************************************/

void StopProgram(void)
{
  DWORD nbr;
  unsigned char ch;

  ch = (char)CMD_STOP_PROGRAM;
  WriteFile( hCom, (LPVOID)(&ch), 1, (LPDWORD)(&nbr), NULL );

  return;
}

/****************************************************************************/

int main( int argc, char *argv[] )
{
  char *portname = "COMX";
  int pn;   // port number
  DWORD br; // baud rate
  int opt;

  if( argc < 2 )
  {
    printf("\nUsage: mc pn\n");
    printf("  pn = serial port number: 1,2,3,4\n");
    return(-1);
  }

// Open the serial port connection to the micro-controler

  pn = atoi( argv[1] );
  if( (pn < 1) || (pn > 4) )
  {
    perror( "Portnumber must be in the range [1,4]\n" );
    return( -1 );
  }
  portname[3] = '0' + pn;

  br = 9600;
  hCom = OpenSerialPort( portname, br );
  if( hCom == NULL )
  {
    perror( "Could not open port\n");
    return( -1 );
  }

  printf( "\nSerial port opened\n" );
  printf( "  Port = %s\n", portname );
  printf( "  Baud = %d\n\n", br );

  do
  {
    printf("==============================================================\n");
    printf("1) Set Program\n");
    printf("2) Get Program\n");
    printf("3) Run Program\n");
    printf("4) Loop Program\n");
    printf("5) Stop Program\n");
    printf("6) Quit\n");
    printf("Enter option [1,6] => ");
    scanf("%d", &opt);
    printf("\n");

    switch( opt )
    {
      case 1:
        SetProgram();
        break;
        
      case 2:
        GetProgram();
        break;

      case 3:
        RunProgram();
        break;

      case 4:
        LoopProgram();
        break;

      case 5:
        StopProgram();
        break;
    }
  }while( opt != 6 );

  CloseHandle(hCom);
  return( 0 );
}
